import { NextRequest, NextResponse } from "next/server";
import { PrismaClient } from "@prisma/client";

const prisma = new PrismaClient();

export async function GET(_req: NextRequest, { params }: { params: { slug: string } }) {
  const link = await prisma.link.findUnique({ where: { slug: params.slug } });
  if (!link) return NextResponse.json({ error: "Not found" }, { status: 404 });
  return NextResponse.json(link);
}

export async function PATCH(req: NextRequest, { params }: { params: { slug: string } }) {
  const body = await req.json().catch(() => ({}));
  const { slug, target, oneTime, expireAt } = body as { slug?: string; target?: string; oneTime?: boolean; expireAt?: string | null };
  const updates: any = {};

  if (typeof target === "string") {
    try { new URL(target); updates.target = target; } catch { return NextResponse.json({ error: "Invalid target URL" }, { status: 400 }); }
  }
  if (typeof oneTime === "boolean") updates.oneTime = oneTime;
  if (expireAt === null) updates.expireAt = null;
  if (typeof expireAt === "string") {
    const d = new Date(expireAt);
    if (isNaN(d.getTime())) return NextResponse.json({ error: "Invalid expireAt" }, { status: 400 });
    updates.expireAt = d;
  }

  // handle slug change (unique)
  if (slug && slug !== params.slug) {
    if (!/^[a-z0-9-]{3,32}$/.test(slug)) return NextResponse.json({ error: "Invalid slug" }, { status: 400 });
    const exist = await prisma.link.findUnique({ where: { slug } });
    if (exist) return NextResponse.json({ error: "Slug already taken" }, { status: 409 });
    // change slug by creating new & deleting old? better to update unique field directly
    try {
      const updated = await prisma.link.update({
        where: { slug: params.slug },
        data: { slug }
      });
      // merge any other updates after slug change
      if (Object.keys(updates).length) {
        await prisma.link.update({ where: { slug }, data: updates });
      }
      return NextResponse.json(await prisma.link.findUnique({ where: { slug } }));
    } catch (e) {
      return NextResponse.json({ error: "Update failed" }, { status: 500 });
    }
  }

  try {
    const updated = await prisma.link.update({ where: { slug: params.slug }, data: updates });
    return NextResponse.json(updated);
  } catch (e) {
    return NextResponse.json({ error: "Update failed" }, { status: 500 });
  }
}

export async function DELETE(_req: NextRequest, { params }: { params: { slug: string } }) {
  try {
    await prisma.link.delete({ where: { slug: params.slug } });
    return NextResponse.json({ ok: true });
  } catch (e) {
    return NextResponse.json({ error: "Delete failed or not found" }, { status: 404 });
  }
}
